<?php
/**
 * API: Exportar Pedidos a Excel
 * Genera un archivo CSV con los productos pedidos según los filtros aplicados
 */

header('Access-Control-Allow-Origin: *');

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/branch.php';

try {
    $pdo = getDBConnection();
    
    // Obtener parámetros de filtro
    $laboratorio = isset($_GET['laboratorio']) ? trim($_GET['laboratorio']) : '';
    $categoria_id = isset($_GET['categoria_id']) ? (int)$_GET['categoria_id'] : 0;
    $sucursal_id = getCurrentBranchId();
    
    // Obtener nombre de la sucursal
    $sqlBranch = "SELECT nombre FROM sucursales WHERE id = ?";
    $stmtBranch = $pdo->prepare($sqlBranch);
    $stmtBranch->execute([$sucursal_id]);
    $sucursal_nombre = $stmtBranch->fetchColumn();
    
    // Construir WHERE clause
    $whereConditions = [];
    $params = [];
    
    if ($sucursal_id) {
        $whereConditions[] = 'p.sucursal_id = ?';
        $params[] = $sucursal_id;
    }

    if (!empty($laboratorio)) {
        $whereConditions[] = 'prod.laboratorio LIKE ?';
        $params[] = "%{$laboratorio}%";
    }

    if ($categoria_id > 0) {
        $whereConditions[] = 'prod.id_categoria = ?';
        $params[] = $categoria_id;
    }
    
    $whereClause = !empty($whereConditions) ? ' WHERE ' . implode(' AND ', $whereConditions) : '';
    
    // Consulta SQL para obtener los datos
    $sql = "SELECT 
                prod.CodigoInterno as codigo_interno,
                dp.nombre_comercial,
                prod.laboratorio,
                c.nombre as categoria,
                dp.cantidad_solicitada,
                dp.presentacion_solicitada
            FROM detalle_pedidos dp
            INNER JOIN pedidos p ON dp.pedido_id = p.id
            INNER JOIN productos prod ON dp.producto_id = prod.id
            LEFT JOIN categorias c ON prod.id_categoria = c.id" . 
            $whereClause .
            " ORDER BY dp.nombre_comercial ASC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $products = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Determinar el nombre del archivo
    $filename = 'Pedidos';
    if (!empty($laboratorio)) {
        $filename = 'Pedidos_' . preg_replace('/[^a-zA-Z0-9]/', '_', $laboratorio);
    } elseif ($categoria_id > 0) {
        // Obtener nombre de categoría
        $sqlCat = "SELECT nombre FROM categorias WHERE id = ?";
        $stmtCat = $pdo->prepare($sqlCat);
        $stmtCat->execute([$categoria_id]);
        $catName = $stmtCat->fetchColumn();
        if ($catName) {
            $filename = 'Pedidos_' . preg_replace('/[^a-zA-Z0-9]/', '_', $catName);
        }
    }
    $filename .= '_' . date('Y-m-d') . '.xls';
    
    // Configurar headers para descarga de Excel
    header('Content-Type: application/vnd.ms-excel; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    // Crear tabla HTML para Excel
    echo '<!DOCTYPE html>';
    echo '<html>';
    echo '<head>';
    echo '<meta charset="UTF-8">';
    echo '<style>';
    echo 'table { border-collapse: collapse; width: 100%; }';
    echo 'th { background-color: #4472C4; color: white; font-weight: bold; padding: 8px; border: 1px solid #000; text-align: left; }';
    echo 'td { padding: 8px; border: 1px solid #000; }';
    echo 'h2 { color: #2E5090; font-family: Arial, sans-serif; }';
    echo '</style>';
    echo '</head>';
    echo '<body>';
    
    // Título con nombre de sucursal y fecha
    $fecha_actual = date('d/m/Y');
    echo '<h2>PEDIDO ' . strtoupper($sucursal_nombre) . ' - ' . $fecha_actual . '</h2>';
    
    // Tabla
    echo '<table>';
    
    // Encabezados
    echo '<tr>';
    echo '<th>Código Interno</th>';
    echo '<th>Nombre</th>';
    echo '<th>Laboratorio</th>';
    echo '<th>Categoría</th>';
    echo '<th>Cantidad</th>';
    echo '<th>Presentación</th>';
    echo '</tr>';
    
    // Datos
    foreach ($products as $product) {
        echo '<tr>';
        echo '<td>' . htmlspecialchars($product['codigo_interno'] ?? '') . '</td>';
        echo '<td>' . htmlspecialchars($product['nombre_comercial'] ?? '') . '</td>';
        echo '<td>' . htmlspecialchars($product['laboratorio'] ?? '') . '</td>';
        echo '<td>' . htmlspecialchars($product['categoria'] ?? 'Sin categoría') . '</td>';
        echo '<td>' . htmlspecialchars($product['cantidad_solicitada'] ?? '') . '</td>';
        echo '<td>' . htmlspecialchars($product['presentacion_solicitada'] ?? '') . '</td>';
        echo '</tr>';
    }
    
    echo '</table>';
    echo '</body>';
    echo '</html>';
    
    exit;
    
} catch (PDOException $e) {
    error_log("Error al exportar pedidos: " . $e->getMessage());
    http_response_code(500);
    echo "Error al generar el archivo";
}
?>
